/************************************************************************
PEFile - A class to manipulate portable executables for Win32
Copyright (C) 2006 Daniel "Lesco" Schoepe

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
************************************************************************/

#include "PEFile.h"

PEFile::PEFile()
{
	filename = 0;
	fileOpened = false;
}

PEFile::PEFile(const char *filename)
{
	PEFile();
	openFile(filename);

}

void PEFile::openFile(const char *filename)
{
	if( fileOpened )
		close();
	if( this->filename )
		delete [] this->filename;
	this->filename = new char[strlen(filename) + 1];
	strcpy(this->filename,filename);
	open();
}

void PEFile::open()
{
	file = CreateFile(filename,GENERIC_READ|GENERIC_WRITE,
			0,0,OPEN_EXISTING,0,0);
	if( file == INVALID_HANDLE_VALUE ) {
		throw (int)(GetLastError());
	}
	fileSize = GetFileSize(file,0);
	if( fileSize < sizeof(IMAGE_NT_HEADERS) + sizeof(IMAGE_DOS_HEADER) ) {
		throw (int)(INVALID_SIZE);
	}
	mapping = CreateFileMapping(file,0,PAGE_READWRITE,0,0,0);
	if( !mapping ) {
		throw (int)(GetLastError());
	}
	mappedFile = (char*)MapViewOfFile(mapping,FILE_MAP_WRITE|FILE_MAP_READ,
			0,0,0);
	if( !mappedFile ) {
		throw (int)(GetLastError());
	}
	//validate pe file:
	dosHeader = (IMAGE_DOS_HEADER*)mappedFile;
	if( dosHeader->e_magic != IMAGE_DOS_SIGNATURE )
		throw (int)(INVALID_MZ_SIG);
	ntHeaders = (IMAGE_NT_HEADERS*)(dosHeader->e_lfanew + mappedFile);
	if( ntHeaders->Signature != IMAGE_NT_SIGNATURE ) 
		throw (int)(INVALID_PE_SIG);
	if( ntHeaders->OptionalHeader.Magic != IMAGE_NT_OPTIONAL_HDR32_MAGIC )
		throw (int)(INVALID_MAGIC);
	sectionTable = (IMAGE_SECTION_HEADER*)
			((DWORD)ntHeaders + sizeof(IMAGE_NT_HEADERS)); 
	fileOpened = true;
}

void PEFile::close()
{
	if( !fileOpened )
		return;
	fileOpened = false;
	UnmapViewOfFile(mappedFile);
	CloseHandle(mapping);
	CloseHandle(file);
}

IMAGE_SECTION_HEADER * PEFile::sectionHeader(int index)
{
	if( !fileOpened ) 
		throw (int)(FILE_NOT_OPENED);
	if( index >= ntHeaders->FileHeader.NumberOfSections ) {
		throw (int)(OUT_OF_BOUNDS);
	}
	else {
		return sectionTable + index;
	}
}

DWORD PEFile::getSize()
{
	if( !fileOpened ) 
		throw (int)(FILE_NOT_OPENED);
	return fileSize;
}

void PEFile::enlargeFile(DWORD bytes)
{
	if( !fileOpened ) 
		throw (int)(FILE_NOT_OPENED);
	setSize(fileSize + bytes);
}

void PEFile::shrinkFile(DWORD bytes)
{
	if( !fileOpened ) 
		throw (int)(FILE_NOT_OPENED);
	if( bytes > fileSize )
		throw (int)(OUT_OF_BOUNDS);
	setSize(fileSize - bytes);
}

void PEFile::setSize(DWORD bytes)
{
	fileSize = bytes;
	UnmapViewOfFile(mappedFile);
	CloseHandle(mapping);
	mapping = CreateFileMapping(file,0,PAGE_READWRITE,0,fileSize,0);
	if( !mapping ) 
		throw (int)(GetLastError());
	mappedFile = (char*)MapViewOfFile(mapping,FILE_MAP_READ|FILE_MAP_WRITE,
				0,0,0);
	if( !mappedFile )
		throw (int)(GetLastError());
}

IMAGE_SECTION_HEADER * PEFile::addSection(DWORD size,char *data)
{
	if( !fileOpened )
		throw (int)(FILE_NOT_OPENED);
	DWORD i,sectionCount,v,headerSize,roundedSize,freeSpace;
	DWORD sizeOfHeaders,fileAlignment;
	IMAGE_SECTION_HEADER *result;
	//often used header values:
	sectionCount = ntHeaders->FileHeader.NumberOfSections;
	sizeOfHeaders = ntHeaders->OptionalHeader.SizeOfHeaders;
	fileAlignment = ntHeaders->OptionalHeader.FileAlignment;
	//determine lowest available VA
	v = 0;
	for( i = 1; i < sectionCount; ++i ) {
		if( sectionTable[i].VirtualAddress >
			sectionTable[v].VirtualAddress )
			v = i;
	}
	roundedSize = roundToFileAlignment(size);
	v = sectionTable[v].VirtualAddress + 
		sectionTable[v].Misc.VirtualSize;
	//determine space between section headers and sections:
	freeSpace = ((DWORD)ntHeaders - (DWORD)mappedFile)
			+ sizeof(IMAGE_NT_HEADERS);
	freeSpace += sectionCount * sizeof(IMAGE_SECTION_HEADER);
	freeSpace = sizeOfHeaders - freeSpace;
	if( freeSpace >= sizeof(IMAGE_SECTION_HEADER) ) {
		//enough space available
		result = sectionTable + sectionCount;
	}
	else {
		//not enough space
		DWORD bufsize = fileSize - sizeOfHeaders;
		char *tmpBuf = new char[bufsize];
		memcpy(tmpBuf,mappedFile + sizeOfHeaders,bufsize);
		enlargeFile(fileAlignment);
		memcpy(mappedFile + sizeOfHeaders + fileAlignment,tmpBuf,
			bufsize);
		delete [] tmpBuf;
		//Update headers
		ntHeaders->OptionalHeader.SizeOfHeaders += fileAlignment;
		ntHeaders->OptionalHeader.SizeOfImage += fileAlignment;
		for( i = 0; i < sectionCount; ++i ) {
			sectionTable[i].PointerToRawData += fileAlignment;
		}
		result = sectionTable + sectionCount;
		
	}
	memset(result,0,sizeof(IMAGE_SECTION_HEADER));
	result->Misc.VirtualSize = roundedSize;
	result->SizeOfRawData = size;
	result->VirtualAddress = v;
	result->PointerToRawData = fileSize;
	result->Characteristics = IMAGE_SCN_MEM_READ |
			IMAGE_SCN_CNT_INITIALIZED_DATA |
			IMAGE_SCN_MEM_WRITE |
			IMAGE_SCN_CNT_UNINITIALIZED_DATA;
	enlargeFile(size);
	memcpy(result->PointerToRawData + mappedFile,
		data,size);
	
	++ntHeaders->FileHeader.NumberOfSections;
	headerSize = (DWORD)ntHeaders - (DWORD)mappedFile;
	headerSize += (sectionCount) * sizeof(IMAGE_SECTION_HEADER);
	headerSize = roundToFileAlignment(headerSize);
	ntHeaders->OptionalHeader.SizeOfHeaders = headerSize;
	ntHeaders->OptionalHeader.SizeOfInitializedData += roundedSize;
	ntHeaders->OptionalHeader.SizeOfUninitializedData += roundedSize;
	ntHeaders->OptionalHeader.SizeOfImage += roundedSize;
	return result;
}

DWORD PEFile::roundToFileAlignment(DWORD val)
{
	if( !fileOpened )
		throw (int)(FILE_NOT_OPENED);
	return roundUp(val,ntHeaders->OptionalHeader.FileAlignment);
}


char * PEFile::formatErrorMessage(int errorcode)
{
	char *result,*msg;
	if( errorcode > 0 ) {
		FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
			FORMAT_MESSAGE_FROM_SYSTEM,0,errorcode,0,
			(char*)&result,0,0);
		return result;
	}
	switch( errorcode ) {
	case INVALID_MZ_SIG:
		msg = "Invalid MZ-Signature";
		break;
	case INVALID_PE_SIG:
		msg = "Invalid PE-Signature";
		break;
	case INVALID_MAGIC:
		msg = "Invalid magic number";
		break;
	case INVALID_SIZE:
		msg = "Invalid size";
		break;
	case OUT_OF_BOUNDS:
		msg = "A parameter is out of bounds";
		break;
	default:
		msg = "An error occured, but I don't know which one";
		break;
		
	}
	//since FormatMessage allocates it's buffers with
	//LocalAlloc and we want the messages to be
	//freeable in the same way, we use LocalAlloc too, instead of new[]
	result = (char*)LocalAlloc(LMEM_ZEROINIT|LMEM_FIXED,
					strlen(msg)+1);
	strcpy(result,msg);
	return result;
}
